/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Ints;
import java.math.RoundingMode;

@GwtCompatible(emulated=true)
public final class IntMath {
    @VisibleForTesting
    static final int MAX_SIGNED_POWER_OF_TWO = 0x40000000;
    @VisibleForTesting
    static final int MAX_POWER_OF_SQRT2_UNSIGNED = -1257966797;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0};
    @VisibleForTesting
    static final int[] powersOf10 = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};
    @VisibleForTesting
    static final int[] halfPowersOf10 = new int[]{3, 31, 316, 3162, 31622, 316227, 3162277, 31622776, 316227766, Integer.MAX_VALUE};
    @VisibleForTesting
    static final int FLOOR_SQRT_MAX_INT = 46340;
    private static final int[] factorials = new int[]{1, 1, 2, 6, 24, 120, 720, 5040, 40320, 362880, 3628800, 39916800, 479001600};
    @VisibleForTesting
    static int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, 65536, 2345, 477, 193, 110, 75, 58, 49, 43, 39, 37, 35, 34, 34, 33};

    @Beta
    public static int ceilingPowerOfTwo(int x11) {
        MathPreconditions.checkPositive("x", x11);
        if (x11 > 0x40000000) {
            throw new ArithmeticException(new StringBuilder(58).append("ceilingPowerOfTwo(").append(x11).append(") not representable as an int").toString());
        }
        return 1 << -Integer.numberOfLeadingZeros(x11 - 1);
    }

    @Beta
    public static int floorPowerOfTwo(int x11) {
        MathPreconditions.checkPositive("x", x11);
        return Integer.highestOneBit(x11);
    }

    public static boolean isPowerOfTwo(int x11) {
        return x11 > 0 & (x11 & x11 - 1) == 0;
    }

    @VisibleForTesting
    static int lessThanBranchFree(int x11, int y11) {
        return ~(~(x11 - y11)) >>> 31;
    }

    public static int log2(int x11, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x11);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(IntMath.isPowerOfTwo(x11));
            }
            case DOWN: 
            case FLOOR: {
                return 31 - Integer.numberOfLeadingZeros(x11);
            }
            case UP: 
            case CEILING: {
                return 32 - Integer.numberOfLeadingZeros(x11 - 1);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Integer.numberOfLeadingZeros(x11);
                int cmp = -1257966797 >>> leadingZeros;
                int logFloor = 31 - leadingZeros;
                return logFloor + IntMath.lessThanBranchFree(cmp, x11);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(int x11, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x11);
        int logFloor = IntMath.log10Floor(x11);
        int floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x11 == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + IntMath.lessThanBranchFree(floorPow, x11);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + IntMath.lessThanBranchFree(halfPowersOf10[logFloor], x11);
            }
        }
        throw new AssertionError();
    }

    private static int log10Floor(int x11) {
        byte y11 = maxLog10ForLeadingZeros[Integer.numberOfLeadingZeros(x11)];
        return y11 - IntMath.lessThanBranchFree(x11, powersOf10[y11]);
    }

    @GwtIncompatible
    public static int pow(int b11, int k11) {
        MathPreconditions.checkNonNegative("exponent", k11);
        switch (b11) {
            case 0: {
                return k11 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k11 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                return k11 < 32 ? 1 << k11 : 0;
            }
            case -2: {
                if (k11 < 32) {
                    return (k11 & 1) == 0 ? 1 << k11 : -(1 << k11);
                }
                return 0;
            }
        }
        int accum = 1;
        while (true) {
            switch (k11) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return b11 * accum;
                }
            }
            accum *= (k11 & 1) == 0 ? 1 : b11;
            b11 *= b11;
            k11 >>= 1;
        }
    }

    @GwtIncompatible
    public static int sqrt(int x11, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x11);
        int sqrtFloor = IntMath.sqrtFloor(x11);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor * sqrtFloor == x11);
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                return sqrtFloor + IntMath.lessThanBranchFree(sqrtFloor * sqrtFloor, x11);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + IntMath.lessThanBranchFree(halfSquare, x11);
            }
        }
        throw new AssertionError();
    }

    private static int sqrtFloor(int x11) {
        return (int)Math.sqrt(x11);
    }

    public static int divide(int p11, int q11, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        if (q11 == 0) {
            throw new ArithmeticException("/ by zero");
        }
        int div = p11 / q11;
        int rem = p11 - q11 * div;
        if (rem == 0) {
            return div;
        }
        int signum = 1 | (p11 ^ q11) >> 31;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int absRem = Math.abs(rem);
                int cmpRemToHalfDivisor = absRem - (Math.abs(q11) - absRem);
                if (cmpRemToHalfDivisor == 0) {
                    increment = mode == RoundingMode.HALF_UP || mode == RoundingMode.HALF_EVEN & (div & 1) != 0;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + signum : div;
    }

    public static int mod(int x11, int m11) {
        if (m11 <= 0) {
            throw new ArithmeticException(new StringBuilder(31).append("Modulus ").append(m11).append(" must be > 0").toString());
        }
        int result = x11 % m11;
        return result >= 0 ? result : result + m11;
    }

    public static int gcd(int a11, int b11) {
        MathPreconditions.checkNonNegative("a", a11);
        MathPreconditions.checkNonNegative("b", b11);
        if (a11 == 0) {
            return b11;
        }
        if (b11 == 0) {
            return a11;
        }
        int aTwos = Integer.numberOfTrailingZeros(a11);
        a11 >>= aTwos;
        int bTwos = Integer.numberOfTrailingZeros(b11);
        b11 >>= bTwos;
        while (a11 != b11) {
            int delta = a11 - b11;
            int minDeltaOrZero = delta & delta >> 31;
            a11 = delta - minDeltaOrZero - minDeltaOrZero;
            b11 += minDeltaOrZero;
            a11 >>= Integer.numberOfTrailingZeros(a11);
        }
        return a11 << Math.min(aTwos, bTwos);
    }

    public static int checkedAdd(int a11, int b11) {
        long result = (long)a11 + (long)b11;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedAdd", a11, b11);
        return (int)result;
    }

    public static int checkedSubtract(int a11, int b11) {
        long result = (long)a11 - (long)b11;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedSubtract", a11, b11);
        return (int)result;
    }

    public static int checkedMultiply(int a11, int b11) {
        long result = (long)a11 * (long)b11;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedMultiply", a11, b11);
        return (int)result;
    }

    public static int checkedPow(int b11, int k11) {
        MathPreconditions.checkNonNegative("exponent", k11);
        switch (b11) {
            case 0: {
                return k11 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k11 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                MathPreconditions.checkNoOverflow(k11 < 31, "checkedPow", b11, k11);
                return 1 << k11;
            }
            case -2: {
                MathPreconditions.checkNoOverflow(k11 < 32, "checkedPow", b11, k11);
                return (k11 & 1) == 0 ? 1 << k11 : -1 << k11;
            }
        }
        int accum = 1;
        while (true) {
            switch (k11) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return IntMath.checkedMultiply(accum, b11);
                }
            }
            if ((k11 & 1) != 0) {
                accum = IntMath.checkedMultiply(accum, b11);
            }
            if ((k11 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-46340 <= b11 & b11 <= 46340, "checkedPow", b11, k11);
            b11 *= b11;
        }
    }

    @Beta
    public static int saturatedAdd(int a11, int b11) {
        return Ints.saturatedCast((long)a11 + (long)b11);
    }

    @Beta
    public static int saturatedSubtract(int a11, int b11) {
        return Ints.saturatedCast((long)a11 - (long)b11);
    }

    @Beta
    public static int saturatedMultiply(int a11, int b11) {
        return Ints.saturatedCast((long)a11 * (long)b11);
    }

    @Beta
    public static int saturatedPow(int b11, int k11) {
        MathPreconditions.checkNonNegative("exponent", k11);
        switch (b11) {
            case 0: {
                return k11 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k11 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                if (k11 >= 31) {
                    return Integer.MAX_VALUE;
                }
                return 1 << k11;
            }
            case -2: {
                if (k11 >= 32) {
                    return Integer.MAX_VALUE + (k11 & 1);
                }
                return (k11 & 1) == 0 ? 1 << k11 : -1 << k11;
            }
        }
        int accum = 1;
        int limit = Integer.MAX_VALUE + (b11 >>> 31 & (k11 & 1));
        while (true) {
            switch (k11) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return IntMath.saturatedMultiply(accum, b11);
                }
            }
            if ((k11 & 1) != 0) {
                accum = IntMath.saturatedMultiply(accum, b11);
            }
            if ((k11 >>= 1) <= 0) continue;
            if (-46340 > b11 | b11 > 46340) {
                return limit;
            }
            b11 *= b11;
        }
    }

    public static int factorial(int n11) {
        MathPreconditions.checkNonNegative("n", n11);
        return n11 < factorials.length ? factorials[n11] : Integer.MAX_VALUE;
    }

    public static int binomial(int n11, int k11) {
        MathPreconditions.checkNonNegative("n", n11);
        MathPreconditions.checkNonNegative("k", k11);
        Preconditions.checkArgument(k11 <= n11, "k (%s) > n (%s)", k11, n11);
        if (k11 > n11 >> 1) {
            k11 = n11 - k11;
        }
        if (k11 >= biggestBinomials.length || n11 > biggestBinomials[k11]) {
            return Integer.MAX_VALUE;
        }
        switch (k11) {
            case 0: {
                return 1;
            }
            case 1: {
                return n11;
            }
        }
        long result = 1L;
        for (int i11 = 0; i11 < k11; ++i11) {
            result *= (long)(n11 - i11);
            result /= (long)(i11 + 1);
        }
        return (int)result;
    }

    public static int mean(int x11, int y11) {
        return (x11 & y11) + ((x11 ^ y11) >> 1);
    }

    @GwtIncompatible
    @Beta
    public static boolean isPrime(int n11) {
        return LongMath.isPrime(n11);
    }

    private IntMath() {
    }
}

