/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Booleans;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.util.Iterator;

@GwtCompatible(emulated=true)
public final class DoubleMath {
    private static final double MIN_INT_AS_DOUBLE = -2.147483648E9;
    private static final double MAX_INT_AS_DOUBLE = 2.147483647E9;
    private static final double MIN_LONG_AS_DOUBLE = -9.223372036854776E18;
    private static final double MAX_LONG_AS_DOUBLE_PLUS_ONE = 9.223372036854776E18;
    private static final double LN_2 = Math.log(2.0);
    @VisibleForTesting
    static final int MAX_FACTORIAL = 170;
    @VisibleForTesting
    static final double[] everySixteenthFactorial = new double[]{1.0, 2.0922789888E13, 2.631308369336935E35, 1.2413915592536073E61, 1.2688693218588417E89, 7.156945704626381E118, 9.916779348709496E149, 1.974506857221074E182, 3.856204823625804E215, 5.5502938327393044E249, 4.7147236359920616E284};

    @GwtIncompatible
    static double roundIntermediate(double x11, RoundingMode mode) {
        if (!DoubleUtils.isFinite(x11)) {
            throw new ArithmeticException("input is infinite or NaN");
        }
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isMathematicalInteger(x11));
                return x11;
            }
            case FLOOR: {
                if (x11 >= 0.0 || DoubleMath.isMathematicalInteger(x11)) {
                    return x11;
                }
                return (long)x11 - 1L;
            }
            case CEILING: {
                if (x11 <= 0.0 || DoubleMath.isMathematicalInteger(x11)) {
                    return x11;
                }
                return (long)x11 + 1L;
            }
            case DOWN: {
                return x11;
            }
            case UP: {
                if (DoubleMath.isMathematicalInteger(x11)) {
                    return x11;
                }
                return (long)x11 + (long)(x11 > 0.0 ? 1 : -1);
            }
            case HALF_EVEN: {
                return Math.rint(x11);
            }
            case HALF_UP: {
                double z11 = Math.rint(x11);
                if (Math.abs(x11 - z11) == 0.5) {
                    return x11 + Math.copySign(0.5, x11);
                }
                return z11;
            }
            case HALF_DOWN: {
                double z12 = Math.rint(x11);
                if (Math.abs(x11 - z12) == 0.5) {
                    return x11;
                }
                return z12;
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int roundToInt(double x11, RoundingMode mode) {
        double z11 = DoubleMath.roundIntermediate(x11, mode);
        MathPreconditions.checkInRangeForRoundingInputs(z11 > -2.147483649E9 & z11 < 2.147483648E9, x11, mode);
        return (int)z11;
    }

    @GwtIncompatible
    public static long roundToLong(double x11, RoundingMode mode) {
        double z11 = DoubleMath.roundIntermediate(x11, mode);
        MathPreconditions.checkInRangeForRoundingInputs(-9.223372036854776E18 - z11 < 1.0 & z11 < 9.223372036854776E18, x11, mode);
        return (long)z11;
    }

    @GwtIncompatible
    public static BigInteger roundToBigInteger(double x11, RoundingMode mode) {
        if (-9.223372036854776E18 - (x11 = DoubleMath.roundIntermediate(x11, mode)) < 1.0 & x11 < 9.223372036854776E18) {
            return BigInteger.valueOf((long)x11);
        }
        int exponent = Math.getExponent(x11);
        long significand = DoubleUtils.getSignificand(x11);
        BigInteger result = BigInteger.valueOf(significand).shiftLeft(exponent - 52);
        return x11 < 0.0 ? result.negate() : result;
    }

    @GwtIncompatible
    public static boolean isPowerOfTwo(double x11) {
        if (x11 > 0.0 && DoubleUtils.isFinite(x11)) {
            long significand = DoubleUtils.getSignificand(x11);
            return (significand & significand - 1L) == 0L;
        }
        return false;
    }

    public static double log2(double x11) {
        return Math.log(x11) / LN_2;
    }

    @GwtIncompatible
    public static int log2(double x11, RoundingMode mode) {
        boolean increment;
        Preconditions.checkArgument(x11 > 0.0 && DoubleUtils.isFinite(x11), "x must be positive and finite");
        int exponent = Math.getExponent(x11);
        if (!DoubleUtils.isNormal(x11)) {
            return DoubleMath.log2(x11 * 4.503599627370496E15, mode) - 52;
        }
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(DoubleMath.isPowerOfTwo(x11));
            }
            case FLOOR: {
                increment = false;
                break;
            }
            case CEILING: {
                increment = !DoubleMath.isPowerOfTwo(x11);
                break;
            }
            case DOWN: {
                increment = exponent < 0 & !DoubleMath.isPowerOfTwo(x11);
                break;
            }
            case UP: {
                increment = exponent >= 0 & !DoubleMath.isPowerOfTwo(x11);
                break;
            }
            case HALF_EVEN: 
            case HALF_UP: 
            case HALF_DOWN: {
                double xScaled = DoubleUtils.scaleNormalize(x11);
                increment = xScaled * xScaled > 2.0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? exponent + 1 : exponent;
    }

    @GwtIncompatible
    public static boolean isMathematicalInteger(double x11) {
        return DoubleUtils.isFinite(x11) && (x11 == 0.0 || 52 - Long.numberOfTrailingZeros(DoubleUtils.getSignificand(x11)) <= Math.getExponent(x11));
    }

    public static double factorial(int n11) {
        MathPreconditions.checkNonNegative("n", n11);
        if (n11 > 170) {
            return Double.POSITIVE_INFINITY;
        }
        double accum = 1.0;
        for (int i11 = 1 + (n11 & 0xFFFFFFF0); i11 <= n11; ++i11) {
            accum *= (double)i11;
        }
        return accum * everySixteenthFactorial[n11 >> 4];
    }

    public static boolean fuzzyEquals(double a11, double b11, double tolerance) {
        MathPreconditions.checkNonNegative("tolerance", tolerance);
        return Math.copySign(a11 - b11, 1.0) <= tolerance || a11 == b11 || Double.isNaN(a11) && Double.isNaN(b11);
    }

    public static int fuzzyCompare(double a11, double b11, double tolerance) {
        if (DoubleMath.fuzzyEquals(a11, b11, tolerance)) {
            return 0;
        }
        if (a11 < b11) {
            return -1;
        }
        if (a11 > b11) {
            return 1;
        }
        return Booleans.compare(Double.isNaN(a11), Double.isNaN(b11));
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(double ... values2) {
        Preconditions.checkArgument(values2.length > 0, "Cannot take mean of 0 values");
        long count = 1L;
        double mean = DoubleMath.checkFinite(values2[0]);
        for (int index = 1; index < values2.length; ++index) {
            DoubleMath.checkFinite(values2[index]);
            mean += (values2[index] - mean) / (double)(++count);
        }
        return mean;
    }

    @Deprecated
    public static double mean(int ... values2) {
        Preconditions.checkArgument(values2.length > 0, "Cannot take mean of 0 values");
        long sum = 0L;
        for (int index = 0; index < values2.length; ++index) {
            sum += (long)values2[index];
        }
        return (double)sum / (double)values2.length;
    }

    @Deprecated
    public static double mean(long ... values2) {
        Preconditions.checkArgument(values2.length > 0, "Cannot take mean of 0 values");
        long count = 1L;
        double mean = values2[0];
        for (int index = 1; index < values2.length; ++index) {
            mean += ((double)values2[index] - mean) / (double)(++count);
        }
        return mean;
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(Iterable<? extends Number> values2) {
        return DoubleMath.mean(values2.iterator());
    }

    @Deprecated
    @GwtIncompatible
    public static double mean(Iterator<? extends Number> values2) {
        Preconditions.checkArgument(values2.hasNext(), "Cannot take mean of 0 values");
        long count = 1L;
        double mean = DoubleMath.checkFinite(values2.next().doubleValue());
        while (values2.hasNext()) {
            double value = DoubleMath.checkFinite(values2.next().doubleValue());
            mean += (value - mean) / (double)(++count);
        }
        return mean;
    }

    @GwtIncompatible
    @CanIgnoreReturnValue
    private static double checkFinite(double argument) {
        Preconditions.checkArgument(DoubleUtils.isFinite(argument));
        return argument;
    }

    private DoubleMath() {
    }
}

